<?PHP if ( ! defined('BASEPATH')) exit('No direct script access allowed');
/**
* @package direct-project-innovation-initiative
* @subpackage models
* @filesource
*//** */

require_once BASEPATH.'core/Model.php';

#TODO - MOVE THIS, PICK BETTER TEXT
/** Default text for something going wrong when instantiating the model */
DEFINE('DEFAULT_ERROR_MESSAGE', 'Please contact an administrator if the problem persists.');


/**
* Parent class for models based on a database table.
* Handles basic CRUD functionality, reducing the amount of custom code that needs to be written for each model & enforcing consistency.
*
* @package direct-project-innovation-initiative
* @subpackage models
*/
class Database_model extends CI_Model {
	protected $table_name;  //name of the table this model is based on; note that it's not named 'table' to avoid colliding with the CI Table library
	protected $primary_key = 'id'; //name of the id field for this table.
	protected $model_alias;
	protected $_field_data; //do not access directly - access via the field_data() method

		
	/**
	* Verifies that a table name and primary key have been set up for this model.
	*/
	public function __construct() {		
		parent::__construct();
		$this->load->database();
        $this->load->library('Validator', array(), 'is');
		
		//default table_name & primary_key to values based off the model name if they're not set up.
		if(empty($this->table_name))
			$this->table_name = plural(strip_from_end('_model', strtolower(get_class($this))));

		//verify that the table_name & primary_key values are valid
		if(!$this->db->table_exists($this->table_name)){
			$this->error->should_be_a_table_in_the_database($this->table_name);
			show_error(DEFAULT_ERROR_MESSAGE);
		}
		if(!$this->field_exists($this->primary_key)){
			$this->error->should_be_a_column_for_table($this->primary_key, $this->table_name);
			show_error(DEFAULT_ERROR_MESSAGE);
		}
		
		$this->model_alias = strip_from_end('_model', strtolower(get_class($this)));
	}		
	
	////////////////////////
	// SEARCH METHODS
	////////////////////////
	
	/**
	* Runs a COUNT query on the table powering this model.
	* Wrapper for CI's {@link http://codeigniter.com/user_guide/database/active_record.html count_all_results()} method.
	*
	* @param int|array Either the id or an array of conditions suitable for CI's {@link http://codeigniter.com/user_guide/database/active_record.html where()} function.
	* @return int
	*/
	public function count($id_or_conditions = array()){
		$this->_set_conditions($id_or_conditions);
		return $this->db->count_all_results($this->table_name);
	}
	
	/**
	* Checks to see if a record exists for this model with the given id or conditions.
	* This will run a count rather than a select, so it's a bit easier on the database then using {@link find} if you don't need the resulting records.
	*
	* @param int|array Either the id or an array of conditions suitable for CI's {@link http://codeigniter.com/user_guide/database/active_record.html where()} function.
	* @return boolean
	*/
	public function exists($id_or_conditions = array()){
		return ($this->count($id_or_conditions) >= 1);
	}
		
	/**
	* Find all records that match the given conditions.
	*
	* @param int|array|string Either a record id, an array of conditions, or a SQL clause 
	* @param string DB column that should be used as the key for the return value.  Defaults to primary key.
	* @return array 
	*/			
	function find($id_or_conditions = array(), $key_by = null){
		if(is_null($key_by)) $key_by = $this->primary_key;
		if(!(is_array($id_or_conditions) && empty($id_or_conditions)) && !$this->_set_conditions($id_or_conditions)) return array(); 
		
		$results = $this->db->get($this->table_name)->result_array();
		if(empty($results)) return $results;	
		if(!is_string($key_by) || !array_key_exists($key_by, first_element($results))){
			if(array_key_exists($this->primary_key, first_element($results))){
				$this->error->should_be_a_column_for_table($key_by, $this->table_name);
				$key_by = $this->primary_key;
			}else{
				$this->error->warning($this->error->describe($key_by).' is not a known key.  The resulting array will not have meaningful keys.');
				return $results;
			}
		}		

		$keyed_results = array();
		foreach($results as $key => $result){
			$keyed_results[$result[$key_by]] = $result;
		}		
		return $keyed_results;
	}
	
	
	/**
	* Finds the first record in the database that matches the given conditions.
	*
	* @param int|array Either the id or an array of conditions suitable for CI's {@link http://codeigniter.com/user_guide/database/active_record.html where()} function.
	* @return array
	*/
	function find_one($id_or_conditions=array()){
		$this->db->limit(1);
		return first_element($this->find($id_or_conditions));
	}
	
	/**
	* Finds a record in the database that matches the conditions, or creates one if none exists.
	*
	* @param array
	* @return array
	*/
	function find_or_create($conditions){
		if(!$this->is->nonempty_array($conditions)) return $this->error->should_be_a_nonempty_array($conditions);
		$record = $this->find_one($conditions);
		if(!is_array($record)) return false; //something went wrong, find_one() triggered an error		
		if(!empty($record)) return $record;
		$record_id = $this->create($conditions);
		if($this->formatted_like_an_id($record_id))
			return $this->find_one($record_id);	
	}
	
	/**
	* Standard way of setting db conditions to be used by all search methods.
	* Note that while this is slightly more limited than CI's {@link http://codeigniter.com/user_guide/database/active_record.html where()} function, 
	* you can always run where() or another Active Record condition method before running one of this model's db functions, and it will be applied to 
	* the next query that is run.
	*
	* @param int|string|array Either an ID or a condition suitable to CI's $db->where() function
	* @return boolean
	*/
	protected function _set_conditions($id_or_conditions=array()){		
		//no conditions set via params
		if(empty($id_or_conditions)) return false;
		
		//searching by id
		if($this->formatted_like_an_id($id_or_conditions)) 
			return $this->db->where($this->primary_key, $id_or_conditions); 
		
		//searching by valid CI conditions
		if((!is_numeric($id_or_conditions) && is_string($id_or_conditions)) || $this->is->nonempty_array($id_or_conditions))
			return $this->db->where($id_or_conditions);
		
		//invalid parameter was passed	
		return $this->error->should_be_an_id_or_a_string_or_an_associative_array($id_or_conditions);
	}
	

	
	//////////////////////////
	// METADATA METHODS
	//////////////////////////
	
	/**
	* Returns metadata about the columns of the table (type, maxlength, etc.)
	* If a field is specified, only metadata about that field will be returned.
	*
	* @param string
	* @return array
	*/
	function field_data($field = null, $property = null){		
		if(empty($this->_field_data)){
			//grab the metadata for this table
			$ci_field_data = $this->db->field_data($this->table_name);
			//grab some metadata that CI doesn't get right
			$db_field_data = $this->db->query("SELECT DATA_TYPE, IS_NULLABLE FROM INFORMATION_SCHEMA.Columns WHERE TABLE_NAME = '".$this->table_name."'")->result_array();	
	
			$field_data = array();		
			foreach($ci_field_data as $key => $metadata){
				$metadata = (array)$metadata;
				unset($metadata['primary_key'], $metadata['default']); //CI doesn't correctly determine these, and thus far I'm not seeing a way to get it from MSSQL
				$metadata['type'] = $db_field_data[$key]['DATA_TYPE'];
				$metadata['required'] = ($db_field_data[$key]['IS_NULLABLE'] == 'NO');
				$this->_field_data[$metadata['name']] = $metadata;
			}
		}
		
		if(!is_null($field)){
			if(!$this->field_exists($field)) return $this->error->should_be_a_column_for_table($field, $this->table_name);
			if(!is_null($property)){
				if(!$this->is->nonempty_string($property)) return $this->error->should_be_a_nonempty_string($property);
				if(!array_key_exists($property, $this->_field_data[$field])) return $this->error->should_be_a_known_field_data_property($property);
				return $this->_field_data[$field][$property];
			}
			return $this->_field_data[$field];
		}
		return $this->_field_data;
	}
	
	/**
	* True if $field is a column on the table for this model.
	* @param string
	* @return boolean
	*/
	function field_exists($field){
		if(!$this->is->nonempty_string($field)) return false;
		return $this->db->field_exists($field, $this->table_name);
	}	 
	
	/**
	* True if $id is formatted like an id for this model.
	* @param scalar
	* @return boolean
	*/
	public function formatted_like_an_id($id){
		return $this->is->nonzero_unsigned_integer($id);
	}

	///////////////////////////
	// DATA MANAGEMENT METHODS
	///////////////////////////
	
	/**
	* Inserts a new record into the database.
	* 
	* This method makes use of a number of hooks that may be overloaded by child classes
	* in order to add validation or custom actions that should take before or after creation.
	*
	* @param array $values Values to be inserted, formatted array($column_name => $value)
	* @return array $record 
	*/	
	function create($values){
		if(!is_array($values)) return $this->error->should_be_an_associative_array($values);
		if(!$this->_run_before_create($values) || !$this->_run_before_create_and_update($values)) 
			return false; //run any specific actions for the child class; quit if actions fail.
		if(!$this->_values_are_valid_for_create($values) || !$this->_values_are_valid_for_create_and_update($values)){
			return $this->error->warning('Unable to create a '.humanize($this->model_alias).' with the following values: '.$this->error->describe($values));
		}

		$id = $this->_create($values);
		if(!$this->formatted_like_an_id($id)){
			return $this->error->warning('Unable to create a '.humanize($this->model_alias).' with the following values: '.$this->error->describe($values));
		}
			
		$record = $this->find_one($id);
		$this->_run_after_create($record);
		$this->_run_after_create_and_update($record);
		return $record;
	}
	
	/**
	* Updates a single record in the database.
	*
	* This method makes use of a number of hooks that may be overloaded by child classes
	* in order to add validation or custom actions that should take before or after the update.
	*
	* @param int $id Primary key for the record
	* @param array $values Array of values to be updated for the record, formatted array($column_name => $value)
	* @return array $record
	*/
	public function update($id, $values){
		if(!$this->formatted_like_an_id($id)) return $this->error->should_be_an_id($id);
		if(!is_array($values)) return $this->error->should_be_an_array($values);

		if(!$this->_run_before_update($id, $values) || !$this->_run_before_create_and_update($values, $id))
			return $this->error->warning('Unable to update '.$this->model_alias.'#'.$id.' with the following values: '.$this->error->describe($values));		
		if(!$this->_values_are_valid_for_update($id, $values) || !$this->_values_are_valid_for_create_and_update($values, $id)){
			return $this->error->warning('Unable to update '.$this->model_alias.'#'.$id.' with the following values: '.$this->error->describe($values));
		}
		if(!$this->_update($id, $values)){
			return $this->error->warning('Unable to update '.$this->model_alias.'#'.$id.' with the following values: '.$this->error->describe($values));
		}
		
		$record = $this->find_one($id);
		$this->_run_after_update($record);
		$this->_run_after_create_and_update($record);
		
		return $record;
	}
	
	/**
	* Deletes a record in the database.
	*
	* This method makes use of hooks that may be overloaded by child classes
	* in order to add custom actions that should take before or after the deletion.
	*
	* @param int $id Primary key for the record
	* @param array $values Array of values to be updated for the record, formatted array($column_name => $value)
	* @return array $record
	*/
	public function delete($id){
		if(!$this->formatted_like_an_id($id)) return $this->error->should_be_an_id($id);
		$record = $this->find_one($id);		
		if(empty($record)){
			return $this->error->should_be_an_x($id, 'id for an existing record for the '.$this->table_name.' table');
		}
			
		if(!$this->_run_before_delete($record)) return false;
		if(!$this->_delete($id)){
			return $this->error->warning('Unable to update '.$this->model_alias.'#'.$id.' with the following values: '.$this->error->describe($values));
		}
		$this->_run_after_delete($record);
		return true;
	}

	
	function _values_are_valid_for_create($values){ return true; }
	
	function _values_are_valid_for_update($id, $values){ return true; }
	
	function _values_are_valid_for_create_and_update($values){ 
		foreach($values as $field => $value){
			if(!$this->field_exists($field)){
				return $this->error->should_be_a_column_for_table($field, $this->table_name);
			}
			$max_length = $this->field_data($field, 'max_length');
			if(!empty($max_length) && strlen($value) > $max_length)
				return $this->error->value_exceeds_max_length_for_table_column($value, $max_length, $field, $this->table_name);		
		}
		return true;
	}
	
	protected function _run_before_create(&$values){
		$now = now(); //we want created_at and modified_at to have the exact same value for this field
		$default_values = array('created_by' => $this->user->id,
								//modified_by will be handled by _run_before_create_and_update
								'created_at' => $now,
								'modified_at' => $now);
		
		//set default values iff this table has the relevant field 						
		foreach($default_values as $field => $value){
			if($this->field_exists($field) && !array_key_exists($field, $values)){
				$values[$field] = $value;
			}
		}
		return true;
	}
	
	/**
	* Hook for child classes to override the actual creation mechanism if necessary.
	* @param array
	* @return int|boolean The id of the new record, or false
	*/
	protected function _create($values){
        $success = $this->db->insert($this->table_name, $values); 
        if(!$success) return false;
        return $this->db->insert_id();
    }
	
	protected function _run_after_create(&$record){}
	
	protected function _run_before_update($id, &$values){ return true; }
	
	protected function _update($id, $values){ return $this->db->update($this->table_name, $values, array($this->primary_key => $id)); }
	
	protected function _run_after_update(&$record){}
	
	protected function _run_before_create_and_update(&$values, $id=null){
		$default_values = array('modified_by' => $this->user->id,
								'modified_at' => now());
		foreach($default_values as $field => $value){
			if($this->field_exists($field) && !array_key_exists($field, $values)){
				$values[$field] = $value;
			}
		}
		return true;
	}
	
	protected function _run_after_create_and_update(&$record){}
	
	protected function _run_before_delete(){ return true; }
	
	protected function _delete($id){
		return $this->db->where($this->primary_key, $id)->delete($this->table_name);
	}
	
	protected function _run_after_delete(){}	
		
}